/* --------------------------------------------------------------
  JsonParser.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

import StyleConfiguration from '../entities/StyleConfiguration'; 
import Collection from '../entities/Collection'; 
import Setting from '../entities/Setting'; 
import BoxSetting from '../entities/BoxSetting';

export default {
	/**
	 * Decode StyleConfiguration JSON String
	 *
	 * @param {Object} jsonStyleConfiguration Object containing the style configuration data.
	 *
	 * @return {StyleConfiguration} Returns the decoded StyleConfiguration object.
	 */
	decodeStyleConfiguration(jsonStyleConfiguration) {
		StyleEdit.Validator.isObject(jsonStyleConfiguration);
		
		const styleConfiguration = new StyleConfiguration();
		
		// Decode the main style configuration properties.
		if (jsonStyleConfiguration.name !== undefined && jsonStyleConfiguration.name !== null) {
			styleConfiguration.setName(jsonStyleConfiguration.name);
		}
		
		if (jsonStyleConfiguration.isActive !== undefined && jsonStyleConfiguration.isActive !== null) {
			styleConfiguration.setActive(jsonStyleConfiguration.isActive);
		}
		
		if (jsonStyleConfiguration.thumbnail !== undefined && jsonStyleConfiguration.thumbnail !== null) {
			styleConfiguration.setThumbnail(jsonStyleConfiguration.thumbnail);
		}
		
		if (jsonStyleConfiguration.boilerplate !== undefined && jsonStyleConfiguration.boilerplate !== null) {
			styleConfiguration.setBoilerplate(jsonStyleConfiguration.boilerplate);
		}
		
		if (jsonStyleConfiguration.creationDate !== undefined && jsonStyleConfiguration.creationDate !== null) {
			styleConfiguration.setCreationDate(new Date(Date.parse(jsonStyleConfiguration.creationDate)));
		}
		
		if (jsonStyleConfiguration.modificationDate !== undefined && jsonStyleConfiguration.modificationDate !== null) {
			styleConfiguration.setModificationDate(new Date(Date.parse(jsonStyleConfiguration.modificationDate)));
		}
		
		if (jsonStyleConfiguration.boilerplateModificationDate !== undefined
			&& jsonStyleConfiguration.boilerplateModificationDate !== null) {
			styleConfiguration.setBoilerplateModificationDate(
				new Date(Date.parse(jsonStyleConfiguration.boilerplateModificationDate))
			);
		}
		
		if (jsonStyleConfiguration.colorPalette !== undefined && jsonStyleConfiguration.colorPalette !== null) {
			styleConfiguration.setColorPalette(jsonStyleConfiguration.colorPalette);
		}
		
		if (jsonStyleConfiguration.customStyles !== undefined && jsonStyleConfiguration.customStyles !== null) {
			styleConfiguration.setCustomStyles(jsonStyleConfiguration.customStyles);
		}
		
		// Decode the settings.
		if (jsonStyleConfiguration.settings !== undefined && jsonStyleConfiguration.settings !== null) {
			const settings = [];
			
			jsonStyleConfiguration.settings.forEach(jsonCollection => {
				const collection = new Collection();
				const collectionEntriesArray = [];
				
				if (jsonCollection.name !== undefined && jsonCollection.name !== null) {
					collection.setName(jsonCollection.name);
				}
				
				if (jsonCollection.type && jsonCollection.type !== null) {
					collection.setType(jsonCollection.type);
				}
				
				if (jsonCollection.icon && jsonCollection.icon !== null) {
					collection.setIcon(jsonCollection.icon);
				}
				
				jsonCollection.entries.forEach(jsonSetting => {
					const setting = (jsonSetting.position) ? new BoxSetting() : new Setting();
					
					if (jsonSetting.group !== undefined && jsonSetting.group !== null) {
						setting.setGroup(jsonSetting.group);
					}
					
					if (jsonSetting.name !== undefined && jsonSetting.name !== null) {
						setting.setName(jsonSetting.name);
					}
					
					if (jsonSetting.type !== undefined && jsonSetting.type !== null) {
						setting.setType(jsonSetting.type);
					}
					
					if (jsonSetting.value !== undefined && jsonSetting.value !== null) {
						setting.setValue(jsonSetting.value);
					}
					
					if (jsonSetting.defaultMetric !== undefined && jsonSetting.defaultMetric !== null) {
						setting.setDefaultMetric(jsonSetting.defaultMetric);
					}
					
					if (jsonSetting.options !== undefined && jsonSetting.options !== null) {
						setting.setOptions(jsonSetting.options);
					}
					
					if (jsonSetting.position !== undefined && jsonSetting.position !== null) {
						setting.setPosition(jsonSetting.position);
					}
					
					if (jsonSetting.tags !== undefined && jsonSetting.tags !== null) {
						setting.setTags(jsonSetting.tags);
					}
					
					collectionEntriesArray.push(setting);
				});
				
				collection.setEntries(collectionEntriesArray);
				
				settings.push(collection);
			});
			
			styleConfiguration.setSettings(settings);
		}
		
		return styleConfiguration;
	},
	
	/**
	 * Encode Style Configuration Instance to JSON String or Object
	 *
	 * @param {StyleConfiguration} styleConfiguration The object to be encoded.
	 * @param {Boolean} encode Optional (true), defines whether to stringify the object or not.
	 *
	 * @return {Object|String} Returns the encoded object or string.
	 */
	encodeStyleConfiguration(styleConfiguration, encode = true) {
		StyleEdit.Validator.isObject(styleConfiguration);
		StyleEdit.Validator.isBool(encode);
		
		const obj = {
			name: styleConfiguration.getName(),
			isActive: styleConfiguration.isActive(),
			thumbnail: styleConfiguration.getThumbnail(),
			boilerplate: styleConfiguration.getBoilerplate(),
			creationDate: styleConfiguration.getCreationDate().toString('yyyy-MM-dd HH:mm:ss'),
			modificationDate: styleConfiguration.getModificationDate().toString('yyyy-MM-dd HH:mm:ss'),
			boilerplateModificationDate: styleConfiguration.getBoilerplateModificationDate().toString('yyyy-MM-dd HH:mm:ss'),
			colorPalette: styleConfiguration.getColorPalette(),
			settings: [],
			customStyles: ''
		};
		
		// Encode settings
		styleConfiguration.getSettings().forEach(settingsCollection => {
			const collection = {};
			
			if (settingsCollection.getName() !== null) {
				collection.name = settingsCollection.getName();
			}
			
			if (settingsCollection.getType() !== null) {
				collection.type = settingsCollection.getType();
			}
			
			if (settingsCollection.getIcon() !== null) {
				collection.icon = settingsCollection.getIcon();
			}
			
			if (settingsCollection.getEntries() !== null) {
				collection.entries = [];
			}
			
			settingsCollection.getEntries().forEach(setting => {
				const entry = {};
				
				if (setting.getGroup() !== null) {
					entry.group = setting.getGroup();
				}
				
				if (setting.getName() !== null) {
					entry.name = setting.getName();
				}
				
				if (setting.getType() !== null) {
					entry.type = setting.getType();
				}
				
				if (setting.getValue() !== null) {
					entry.value = setting.getValue();
				}
				
				if (setting.getDefaultMetric() !== null) {
					entry.defaultMetric = setting.getDefaultMetric();
				}
				
				if (setting.getOptions() !== null) {
					entry.options = setting.getOptions();
				}
				
				if (setting.getTags() !== null) {
					entry.tags = setting.getTags();
				}
				
				if (setting.getPosition !== undefined) {
					entry.position = setting.getPosition();
				}
				
				collection.entries.push(entry);
			});
			
			obj.settings.push(collection);
		});
		
		// Encode customStyles
		obj.customStyles = styleConfiguration.getCustomStyles();
		
		return (encode) ? JSON.stringify(obj) : obj;
	}
}
